package data.scripts.campaign.missions;

import com.fs.starfarer.api.Global;
import com.fs.starfarer.api.campaign.BaseOnMessageDeliveryScript;
import com.fs.starfarer.api.campaign.BattleAPI;
import com.fs.starfarer.api.campaign.CampaignFleetAPI;
import com.fs.starfarer.api.campaign.CargoAPI;
import com.fs.starfarer.api.campaign.FactionAPI;
import com.fs.starfarer.api.campaign.FleetAssignment;
import com.fs.starfarer.api.campaign.InteractionDialogAPI;
import com.fs.starfarer.api.campaign.LocationAPI;
import com.fs.starfarer.api.campaign.RepLevel;
import com.fs.starfarer.api.campaign.SectorEntityToken;
import com.fs.starfarer.api.campaign.StarSystemAPI;
import com.fs.starfarer.api.campaign.comm.CommMessageAPI;
import com.fs.starfarer.api.campaign.comm.MessagePriority;
import com.fs.starfarer.api.campaign.econ.MarketAPI;
import com.fs.starfarer.api.campaign.events.CampaignEventTarget;
import com.fs.starfarer.api.characters.FullName.Gender;
import com.fs.starfarer.api.characters.PersonAPI;
import com.fs.starfarer.api.combat.BattleCreationContext;
import com.fs.starfarer.api.fleet.FleetMemberAPI;
import com.fs.starfarer.api.fleet.FleetMemberType;
import com.fs.starfarer.api.impl.campaign.FleetEncounterContext;
import com.fs.starfarer.api.impl.campaign.FleetInteractionDialogPluginImpl.FIDConfig;
import com.fs.starfarer.api.impl.campaign.FleetInteractionDialogPluginImpl.FIDConfigGen;
import com.fs.starfarer.api.impl.campaign.FleetInteractionDialogPluginImpl.FIDDelegate;
import com.fs.starfarer.api.impl.campaign.events.BaseEventPlugin;
import com.fs.starfarer.api.impl.campaign.events.OfficerManagerEvent;
import com.fs.starfarer.api.impl.campaign.events.OfficerManagerEvent.SkillPickPreference;
import com.fs.starfarer.api.impl.campaign.fleets.FleetFactory;
import com.fs.starfarer.api.impl.campaign.fleets.FleetFactoryV2;
import com.fs.starfarer.api.impl.campaign.fleets.FleetParams;
import com.fs.starfarer.api.impl.campaign.ids.Abilities;
import com.fs.starfarer.api.impl.campaign.ids.Factions;
import com.fs.starfarer.api.impl.campaign.ids.MemFlags;
import com.fs.starfarer.api.impl.campaign.ids.Ranks;
import com.fs.starfarer.api.impl.campaign.ids.Strings;
import com.fs.starfarer.api.impl.campaign.ids.Tags;
import com.fs.starfarer.api.loading.FleetCompositionDoctrineAPI;
import com.fs.starfarer.api.util.IntervalUtil;
import com.fs.starfarer.api.util.Misc;
import com.fs.starfarer.api.util.WeightedRandomPicker;
import data.scripts.SWPModPlugin;
import data.scripts.campaign.SWP_FleetFactory;
import data.scripts.campaign.SWP_FleetFactory.FleetFactoryDelegate;
import data.scripts.campaign.events.SWP_IBBTracker;
import data.scripts.util.SWP_Util;
import data.scripts.util.SWP_Util.RequiredFaction;
import java.awt.Color;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import org.apache.log4j.Logger;
import org.lazywizard.lazylib.MathUtils;

public class SWP_FamousBountyEvent extends BaseEventPlugin {

    public static final Set<String> BANNED_SYSTEMS = new HashSet<>(1);
    public static final Set<String> IBB_FACTIONS = new LinkedHashSet<>(6);
    public static final int MAX_BOUNTY_STAGE = FamousBountyStage.values().length - 1;
    public static Logger log = Global.getLogger(SWP_FamousBountyEvent.class);

    static {
        IBB_FACTIONS.add(Factions.HEGEMONY);
        IBB_FACTIONS.add(Factions.PERSEAN);
        IBB_FACTIONS.add("interstellarimperium");
        IBB_FACTIONS.add("citadeldefenders");
        IBB_FACTIONS.add("shadow_industry");
        IBB_FACTIONS.add("tiandong");
        IBB_FACTIONS.add("ORA");
    }

    public static int calculatePowerLevel(FamousBountyStage stage) {
        float pointFactor = 5f * (stage.qf / 4f + 0.875f);
        if (stage.fleetFaction.contentEquals(Factions.TRITACHYON)) {
            pointFactor *= 1.2f;
        }
        if (stage.fleetFaction.contentEquals(Factions.REMNANTS)) {
            pointFactor *= 1.1f;
        }
        if (stage.fleetFaction.contentEquals(Factions.LIONS_GUARD)) {
            pointFactor *= 1.3f;
        }
        if (stage.fleetFaction.contentEquals(Factions.PIRATES)) {
            pointFactor *= 0.75f;
        }
        if (stage.fleetFaction.contentEquals(Factions.DERELICT)) {
            pointFactor *= 0.6f;
        }
        if (stage.fleetFaction.contentEquals("shadow_industry")) {
            pointFactor *= 1.1f;
        }
        if (stage.fleetFaction.contentEquals("domain")) {
            pointFactor *= 1.1f;
        }
        if (stage.fleetFaction.contentEquals("sector")) {
            pointFactor *= 1.2f;
        }
        if (stage.fleetFaction.contentEquals("everything")) {
            pointFactor *= 1.3f;
        }
        if (stage.fleetFaction.contentEquals("templars")) {
            pointFactor *= 2f;
        }
        if (stage.fleetFaction.contentEquals("cabal")) {
            pointFactor *= 1.4f;
        }
        if (stage.fleetFaction.contentEquals("diableavionics")) {
            pointFactor *= 1.1f;
        }
        if (stage.fleetFaction.contentEquals("ORA")) {
            pointFactor *= 1.2f;
        }
        pointFactor *= (1f + stage.opBonus / 100f);
        int power = Math.round((stage.maxPts + Math.round(stage.ptsFromBoss * 1.5f)) * pointFactor);
        int offLvl = stage.officerCount * stage.officerLevel;
        int cdrLvl = stage.level;
        power *= cdrLvl / 200f + 1f;
        power += cdrLvl + offLvl;
        return power;
    }

    public static MarketAPI findNearestMarket(SectorEntityToken token, float maxDist) {
        List<MarketAPI> localMarkets = Misc.getMarketsInLocation(token.getContainingLocation());
        float distToLocalMarket = Float.MAX_VALUE;
        MarketAPI closest = null;
        for (MarketAPI market : localMarkets) {
            if (market.isPlanetConditionMarketOnly()) {
                continue;
            }
            if (market.getPrimaryEntity() == null) {
                continue;
            }
            if (market.getPrimaryEntity().getContainingLocation() != token.getContainingLocation()) {
                continue;
            }

            float currDist = Misc.getDistance(market.getPrimaryEntity().getLocation(), token.getLocation());
            if (currDist > maxDist) {
                continue;
            }
            if (currDist < distToLocalMarket) {
                distToLocalMarket = currDist;
                closest = market;
            }
        }
        return closest;
    }

    private float bountyCredits = 0;
    private FactionAPI bountyFaction = null;
    private final IntervalUtil checkInterval = new IntervalUtil(0.5f, 1f);
    private boolean completed = false;
    private final float duration = 180f;
    private float elapsedDays = 0f;
    private boolean ended = false;
    private FleetMemberAPI flagship;
    private String flagshipName;
    private CampaignFleetAPI fleet;
    private SectorEntityToken hideoutLocation = null;
    private boolean it = false;
    private MessagePriority messagePriority;
    private float payment = 0;
    private PersonAPI person;
    private String targetDesc;
    private String targetDescLong;
    private FamousBountyStage thisStage;

    @Override
    public void advance(float amount) {
        if (!isEventStarted()) {
            return;
        }
        if (fleet != null && fleet.isAlive()) {
            if (fleet.isInCurrentLocation() && fleet.getFaction() != bountyFaction) {
                fleet.setFaction(bountyFaction.getId());
            } else if (!fleet.isInCurrentLocation() && !fleet.getFaction().getId().contentEquals(Factions.NEUTRAL)) {
                fleet.setFaction(Factions.NEUTRAL);
            }
        }
        if (isDone()) {
            return;
        }

        float days = Global.getSector().getClock().convertToDays(amount);
        elapsedDays += days;

        checkInterval.advance(days);
        if (checkInterval.intervalElapsed()) {
            for (FactionAPI fac : Global.getSector().getAllFactions()) {
                if (fac != Global.getSector().getFaction(Factions.PLAYER)) {
                    fac.setRelationship("famous_bounty", RepLevel.INHOSPITABLE);
                }
            }

            boolean freeToDespawn = true;
            if (Global.getSector().getPlayerFleet() != null && fleet != null) {
                if (Global.getSector().getPlayerFleet().getContainingLocation() == fleet.getContainingLocation()) {
                    freeToDespawn = false;
                }
            }

            if (elapsedDays >= duration && !isDone() && freeToDespawn) {
                Global.getSector().reportEventStage(this, "other_end", market.getPrimaryEntity(), messagePriority);
                endEvent(false);
                return;
            }
        }

        if (fleet.getFlagship() == null || fleet.getFlagship().getCaptain() != person) {
            Global.getSector().reportEventStage(this, "other_end", market.getPrimaryEntity(), messagePriority);
            endEvent(false);
        }
    }

    public void endEvent(boolean expire) {
        ended = true;
        if (expire) {
            SWP_IBBTracker.getTracker().reportStageExpired(thisStage);
        } else {
            SWP_IBBTracker.getTracker().reportStageCompleted(thisStage);
        }

        if (fleet != null) {
            fleet.getMemoryWithoutUpdate().set("$stillAlive", false);
            fleet.getMemoryWithoutUpdate().unset(MemFlags.MEMORY_KEY_MAKE_AGGRESSIVE);
            fleet.getMemoryWithoutUpdate().set(MemFlags.MEMORY_KEY_MAKE_NON_AGGRESSIVE, true);
            fleet.getMemoryWithoutUpdate().set(MemFlags.MEMORY_KEY_MAKE_ALLOW_DISENGAGE, true);
            Misc.makeUnimportant(fleet, "swp_ibb");
            fleet.clearAssignments();
            if (hideoutLocation != null) {
                fleet.getAI().addAssignment(FleetAssignment.GO_TO_LOCATION_AND_DESPAWN, hideoutLocation, 1000000f, null);
            } else {
                fleet.despawn();
            }
        }
    }

    @Override
    public String getCurrentImage() {
        return person.getPortraitSprite();
    }

    @Override
    public String getCurrentMessageIcon() {
        return "graphics/swp/icons/intel/swp_famousbounty.png";
    }

    @Override
    public CampaignEventCategory getEventCategory() {
        return CampaignEventCategory.MISSION;
    }

    @Override
    public String getEventIcon() {
        return "graphics/swp/icons/intel/swp_famousbounty.png";
    }

    @Override
    public String getEventName() {
        int daysLeft = (int) (duration - elapsedDays);
        String days;
        if (daysLeft > 0) {
            days = ", " + daysLeft + "d";
        } else {
            days = ", <1d";
        }

        if (isDone()) {
            if (completed) {
                return "IBB: " + person.getName().getFullName() + " - completed";
            } else {
                return "IBB: " + person.getName().getFullName() + " - over";
            }
        }
        return "IBB: " + person.getName().getFullName() + " - " + Misc.getWithDGS(bountyCredits) + Strings.C + days;
    }

    @Override
    public Color[] getHighlightColors(String stageId) {
        List<Color> result = new ArrayList<>(3);
        result.add(Global.getSettings().getColor("buttonShortcut"));

        if (stageId != null && stageId.startsWith("start_")) {
            result.add(Global.getSettings().getColor("buttonShortcut"));
        }

        if (Global.getSector().getPlayerFleet() != null && thisStage != null) {
            int powerLevel = SWP_Util.calculatePowerLevel(Global.getSector().getPlayerFleet());
            int dangerLevel = calculatePowerLevel(thisStage);
            if (powerLevel < Math.round(dangerLevel * 0.25f)) {
                result.add(new Color(255, 0, 0));
            } else if (powerLevel < Math.round(dangerLevel * 0.5f)) {
                result.add(new Color(255, 85, 0));
            } else if (powerLevel < Math.round(dangerLevel * 0.75f)) {
                result.add(new Color(255, 170, 0));
            } else if (powerLevel < Math.round(dangerLevel * 1f)) {
                result.add(new Color(255, 255, 0));
            } else if (powerLevel < Math.round(dangerLevel * 1.5f)) {
                result.add(new Color(170, 255, 0));
            } else if (powerLevel < Math.round(dangerLevel * 2f)) {
                result.add(new Color(85, 255, 0));
            } else {
                result.add(new Color(0, 255, 0));
            }
        }

        return result.toArray(new Color[result.size()]);
    }

    @Override
    public String[] getHighlights(String stageId) {
        List<String> result = new ArrayList<>(3);
        addTokensToList(result, "$bountyCredits");

        if (stageId != null && stageId.startsWith("start_")) {
            addTokensToList(result, "$daysLeft");
        }

        if (Global.getSector().getPlayerFleet() != null && thisStage != null) {
            addTokensToList(result, "$dangerLevel");
        }

        return result.toArray(new String[result.size()]);
    }

    @Override
    public void setParam(Object param) {
        market = (MarketAPI) param;
    }

    public FamousBountyStage getStage() {
        return thisStage;
    }

    @Override
    public Map<String, String> getTokenReplacements() {
        Map<String, String> map = super.getTokenReplacements();
        // this is ok because the token replacement happens right as the message is sent, not when it's received
        // so the lastBounty is the correct value for the message
        map.put("$bountyCredits", Misc.getDGSCredits(payment));
        map.put("$sender", "International Bounty Board");

        map.put("$daysLeft", "" + (int) +Math.max(1, duration - elapsedDays));
        map.put("$targetName", person.getName().getFullName());
        map.put("$targetRank", person.getRank());

        if (Global.getSector().getPlayerFleet() != null && thisStage != null) {
            int powerLevel = SWP_Util.calculatePowerLevel(Global.getSector().getPlayerFleet());
            int dangerLevel = calculatePowerLevel(thisStage);
            String dangerString;
            if (powerLevel < Math.round(dangerLevel * 0.25f)) {
                dangerString = "Extreme";
            } else if (powerLevel < Math.round(dangerLevel * 0.5f)) {
                dangerString = "Very High";
            } else if (powerLevel < Math.round(dangerLevel * 0.75f)) {
                dangerString = "High";
            } else if (powerLevel < Math.round(dangerLevel * 1f)) {
                dangerString = "Moderate";
            } else if (powerLevel < Math.round(dangerLevel * 1.5f)) {
                dangerString = "Low";
            } else if (powerLevel < Math.round(dangerLevel * 2f)) {
                dangerString = "Very Low";
            } else {
                dangerString = "None";
            }
            map.put("$dangerLevel", dangerString);
        }

        if (targetDesc != null) {
            map.put("$targetDesc", targetDesc);
        }
        if (targetDescLong != null) {
            map.put("$targetDescLong", targetDescLong);
        }

        if (bountyFaction != null) {
            map.put("$targetFaction", bountyFaction.getDisplayName());
            map.put("$theTargetFaction", bountyFaction.getDisplayNameWithArticle());
            map.put("$targetFactionLong", bountyFaction.getDisplayNameLong());
            map.put("$theTargetFactionLong", bountyFaction.getDisplayNameLongWithArticle());
        }

        if (hideoutLocation != null) {
            if (hideoutLocation.getContainingLocation() instanceof StarSystemAPI) {
                map.put("$hideoutSystem", ((StarSystemAPI) hideoutLocation.getContainingLocation()).getBaseName());
            } else {
                map.put("$hideoutSystem", hideoutLocation.getContainingLocation().getName());
            }
            map.put("$hideout", hideoutLocation.getName());
        }

        if (SWPModPlugin.isExerelin) {
            map.put("$memberFactions", "many factions");
        } else {
            List<String> memberFactions = new ArrayList<>(SWP_FamousBountyEvent.IBB_FACTIONS.size());
            for (String memberFacId : SWP_FamousBountyEvent.IBB_FACTIONS) {
                FactionAPI memberFac;
                try {
                    memberFac = Global.getSector().getFaction(memberFacId);
                } catch (Exception e) {
                    continue;
                }
                if (memberFac != null) {
                    memberFactions.add(memberFac.getDisplayNameWithArticle());
                }
            }
            map.put("$memberFactions", Misc.getAndJoined(memberFactions.toArray(new String[memberFactions.size()])));
        }

        return map;
    }

    @Override
    public void init(String type, CampaignEventTarget eventTarget) {
        super.init(type, eventTarget);

        thisStage = SWP_IBBTracker.getTracker().getLowestIncompleteStage();
        if (thisStage == null) {
            ended = true;
            return;
        }

        if ((SWP_IBBTracker.getTracker().isStagePosted(thisStage)
                || SWP_IBBTracker.getTracker().isStageBegun(thisStage)) && !SWPModPlugin.Param_LegacyIBB) {
            float powerLevel = SWP_Util.calculatePowerLevel(Global.getSector().getPlayerFleet());
            int matchingStageNum = 0;
            for (int i = 0; i < FamousBountyStage.values().length; i++) {
                FamousBountyStage stage = SWP_IBBTracker.getStage(i);
                if (SWP_FamousBountyEvent.calculatePowerLevel(stage) <= powerLevel) {
                    matchingStageNum = i;
                } else {
                    break;
                }
            }

            int stageNum;
            int currentStageNum = thisStage.ordinal();
            if (matchingStageNum <= currentStageNum) {
                stageNum = currentStageNum + Math.round(Math.abs((float) (new Random()).nextGaussian() * 2f));
            } else {
                stageNum = Math.max(MathUtils.getRandomNumberInRange(currentStageNum, matchingStageNum),
                        currentStageNum + Math.round(Math.abs((float) (new Random()).nextGaussian() * 2f)));
            }
            stageNum = Math.max(currentStageNum, stageNum);

            while (stageNum > currentStageNum) {
                FamousBountyStage stage = SWP_IBBTracker.getStage(stageNum);
                if (SWP_IBBTracker.getTracker().isStageAvailable(stage)) {
                    break;
                } else {
                    stageNum--;
                }
            }

            thisStage = SWP_IBBTracker.getStage(stageNum);
            if (thisStage == null) {
                ended = true;
                return;
            }
        }

        if (!SWP_IBBTracker.getTracker().isStageAvailable(thisStage)) {
            ended = true;
            return;
        }

        messagePriority = MessagePriority.SECTOR;
        bountyFaction = Global.getSector().getFaction("famous_bounty");
        person = initPerson(bountyFaction);

        if (thisStage == null) {
            ended = true;
            return;
        }

        bountyCredits = thisStage.reward;
        payment = bountyCredits;

        it = thisStage.machine;
        switch (thisStage) {
            case STAGE_UNDINE: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 2 mod-frigates.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", is known to possess a small, rag-tag fleet, comprised of a number of renegade pirates. The primary danger is "
                        + person.getName().getLast()
                        + "'s pair of unique Lashers, which " + heOrShe() + " has put to use raiding civilian supply lines.";
                break;
            }
            case STAGE_HADES: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 4 mod-freighters.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", is known to possess a small converted trade fleet, largely purchased privately. "
                        + person.getName().getLast()
                        + " is in possession of four unique Cerberuses, which " + heOrShe()
                        + " stole from a private collector several months ago.";
                break;
            }
            case STAGE_PONY: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 2 mod-freighters.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", is known to possess a private flotilla, helmed by a pair of unique ships. One of these is a unique Mule, which "
                        + person.getName().getLast()
                        + " recently used to steal the other, a unique Tarsus. " + heOrSheCapital()
                        + " is a dangerous criminal who will stop at nothing to bring chaos to the Sector.";
                break;
            }
            case STAGE_ELDER_ORB: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-destroyer.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", is wanted dead for arson, murder, and invasion of privacy. " + heOrSheCapital()
                        + " is a dangerous sexual deviant, known to perform bizarre and horrifying experiments on those "
                        + heOrShe()
                        + " captures. " + person.getName().getLast()
                        + " currently owns a unique Beholder and has a large fleet at " + hisOrHer() + " beck and call.";
                break;
            }
            case STAGE_FRACTURE: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 2 mod-destroyers.";
                targetDescLong = "The target, former Lion's Guard officer " + person.getName().getFirst() + " "
                        + person.getName().getLast()
                        + ", officially declared war against \"the Sector\". As a result, " + person.getName().getLast()
                        + " is now wanted dead by all governments. "
                        + heOrSheCapital() + " possesses a unique Sunder and a unique Hammerhead, making " + himOrHer()
                        + " a dangerous foe.";
                break;
            }
            case STAGE_LIBERTY: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-cruiser.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", was once an esteemed leader. Now, "
                        + heOrShe() + " has lost " + hisOrHer()
                        + " sanity, carrying on against the Sector in the name of a fictional empire. "
                        + person.getName().getLast() + " commands a unique Wuzhang, and leads a band of indoctrinated cronies.";
                break;
            }
            case STAGE_RAPTOR: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-cruiser and 1 mod-destroyer.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", is the daughter of a retired admiral. Using the privilege " + heOrShe() + " received, "
                        + person.getName().getLast()
                        + " acquired a unique Falcon and a unique Medusa. Now, " + person.getName().getLast()
                        + " is on a romp of destruction, destroying every fleet "
                        + heOrShe() + " can catch.";
                break;
            }
            case STAGE_BULLSEYE: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 2 upgraded battleships, 4 upgraded destroyers, and 4 upgraded frigates.";
                targetDescLong = "The target, P.A.C.K admiral " + person.getName().getFirst() + " "
                        + person.getName().getLast()
                        + ", was purportedly spliced with canine genes by " + hisOrHer() + " own request. While "
                        + person.getName().getLast()
                        + " did survive the operation, " + heOrShe()
                        + " went partially insane, swiping a group of newly-modded ships and proceeding to terrorize the space lanes. Be advised: "
                        + person.getName().getLast() + "'s fleet is mostly comprised of unique P.A.C.K. warships; however, "
                        + person.getName().getLast()
                        + " is known not to possess any mod-ships. Adjust your cost evaluations accordingly.";
                break;
            }
            case STAGE_APEX: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-cruiser.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", was a famed war hero, known for singlehandedly stopping the Rebellion of Byth. However, " + heOrShe()
                        + " later fell victim to schizophrenia, eventually summoning the most loyal members of " + hisOrHer()
                        + " old fleet and launching on a rampage against the Sector. Be advised: despite "
                        + person.getName().getLast() + "'s mental disorder, "
                        + heOrShe() + " remains an extremely skilled pilot and commander, and is in possession of a unique Eagle.";
                break;
            }
            case STAGE_EMPEROR: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-cruiser and 4 mod-gunships.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", was a double-agent, working for the Tri-Tachyon Corporation in secret while posing as a Hegemon commander. "
                        + heOrSheCapital()
                        + " eventually betrayed the Tri-Tachyon, but in the process revealed " + hisOrHer()
                        + " treason to the Hegemony. Now, "
                        + person.getName().getLast() + " is wanted dead by both factions. " + heOrSheCapital()
                        + " still possesses " + hisOrHer()
                        + " loyal fleet, and has a unique Dominator as well as four unique Brawlers under " + hisOrHer()
                        + " command.";
                break;
            }
            case STAGE_GULF: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-cruiser.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", is wanted on charges of treason by the Diable Corporation."
                        + " Under normal circumstances, the International Bounty Board would not endorse the Diable Corporation. However, "
                        + person.getName().getLast()
                        + " has carved a war-path through allied territory, causing untold damage to local infrastructure. As such, "
                        + heOrShe()
                        + " is officially wanted dead by all IBB member polities. " + person.getName().getLast()
                        + " commands a large Diable fleet from " + hisOrHer()
                        + " unique Gust.";
                break;
            }
            case STAGE_FRAMEBREAKER: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-battleship and 2 mod-cruisers.";
                targetDescLong = "The target, " + person.getName().getFirst() + " \"King\" "
                        + person.getName().getLast() + ", fashioned " + himOrHer()
                        + "self after the legendary prophet Ludd, acquiring a large fleet of followers. " + hisOrHerCapital()
                        + " rampage has left full military patrols decimated, escalating " + person.getName().getLast()
                        + " to the IBB list. " + "King "
                        + person.getName().getLast() + " commands a fleet of fanatics from a unique Onslaught. "
                        + heOrSheCapital()
                        + " cronies are also in possession of a pair of unique Dominators.";
                break;
            }
            case STAGE_LEVIATHAN: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-superfreighter and 2 mod-tankers.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", was a multi-millionaire trade magnate. However, " + heOrShe()
                        + " recently fell afoul of several governments after being exposed for exploiting food shortages and trading on the black market. "
                        + person.getName().getLast()
                        + " is currently on the run, commanding a large privateer armada from a massive unique Atlas. "
                        + heOrSheCapital()
                        + " is also in possession of a pair of unique Phaetons.";
                break;
            }
            case STAGE_POSEIDON: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-carrier.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", is a known terrorist who has been at large for several years. Now, the IBB has placed a bounty on "
                        + hisOrHer()
                        + " head, inviting mercenaries and bounty hunters to take this dangerous individual down. Along with "
                        + hisOrHer() + " large fleet, "
                        + person.getName().getLast() + " owns a unique Charybdis, which " + heOrShe()
                        + " reportedly plundered from a Shadowyards fleet " + heOrShe()
                        + " destroyed in the past.";
                break;
            }
            case STAGE_ILIAD: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-battlecruiser and 1 mod-cruiser.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", is a well-known fleet commander, famously wheelchair-bound due to injuries sustained during "
                        + hisOrHer()
                        + " final battle. Now, " + heOrShe() + " seeks to regain " + hisOrHer() + " former glory by taking "
                        + hisOrHer()
                        + " armada on a crash run through the sector, causing as much damage as possible. "
                        + person.getName().getLast() + " commands " + hisOrHer()
                        + " signature unique Odyssey and unique Aurora, making " + hisOrHer() + " fleet extremely dangerous.";
                break;
            }
            case STAGE_OLYMPUS_X: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-missile platform and 8 mod-destroyers.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", went rogue for unknown reasons. " + hisOrHerCapital()
                        + " now leads a large Imperial armada, proceeding on missions of mass destruction. "
                        + person.getName().getLast() + " must be destroyed as soon as possible before " + hisOrHer()
                        + " unique Olympus can cause any more permanent environmental damage. Be advised: "
                        + person.getName().getLast()
                        + "'s fleet also includes eight unique Praetorians.";
                break;
            }
            case STAGE_FRANKENSTEIN: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-cruiser.";
                targetDescLong = "The target, Sir " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", inherited a tremendous fortune from " + hisOrHer() + " father, including a unique hull vaguely "
                        + "describable as an unholy amalgamation of dead ships. While " + person.getName().getLast()
                        + " was once considered a respected knight, " + heOrShe() + " is currently on a rampaging joyride, "
                        + "causing countless deaths along " + hisOrHer()
                        + " path of destruction. Warning: a large fleet of well-outfitted pirate vessels follows the devastation.";
                break;
            }
            case STAGE_LUCIFER: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet comprised of 2 phase mod-cruisers, 4 phase mod-destroyers, and 8 phase mod-frigates.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", is a failed Tri-Tachyon Alpha+ AI experiment. Quickly falling rampant, " + heOrShe()
                        + " stole the plans for a prototype phase coil. Later, it was discovered that "
                        + person.getName().getLast()
                        + " put this technology to use, creating a powerful fleet consisting entirely of unique phase ships. "
                        + hisOrHerCapital()
                        + " continued existence has been declared a priority-alpha security vulnerability by the IBB.";
                break;
            }
            case STAGE_ODIN: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-battlecruiser and 6 mod-destroyers.";
                targetDescLong = "The target is an artificial entity, designated " + person.getName().getFirst() + " "
                        + person.getName().getLast()
                        + ". Being an ancient artificial intelligence, " + heOrShe()
                        + " predates the Collapse by an unknown period of time. Lately, "
                        + person.getName().getLast()
                        + " has gone rampant, acquiring a large fleet of hodge-podge ships. For reasons of international security,"
                        + " this AI must be destroyed as soon as possible. " + person.getName().getLast()
                        + " was last seen controlling a large fleet of mixed ships, including a unique Mimir and six unique Potnia-bisses.";
                break;
            }
            case STAGE_EUPHORIA: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-carrier and 6 mod-fighter wings.";
                targetDescLong = "The target, famed drug magnate " + person.getName().getFirst() + " "
                        + person.getName().getLast()
                        + ", is wanted on charges of terrorism and crimes against humanity. " + heOrSheCapital()
                        + " is known to have ransacked Tri-Tachyon drug production centers, killing thousands of low-level workers and wrecking the economy of at least"
                        + " one core world. " + person.getName().getLast()
                        + " is also known to have connections to the Starlight Cabal, having acquired a powerful fleet of high-end warships. Exercise extreme caution"
                        + " when engaging " + person.getName().getLast() + ", as " + heOrShe()
                        + " was last seen commanding a unique Astral and six wings of unique Wasp fighters.";
                break;
            }
            case STAGE_CLERIC: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-retributor.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", is an estranged Templar leader. While this would normally be cause for celebration, it seems that "
                        + hisOrHer()
                        + " alignment is still strictly anti-Domain. " + person.getName().getLast()
                        + " seeks to destroy everything " + heOrShe()
                        + " can, so the IBB has placed an unprecedented bounty on " + person.getName().getLast()
                        + "'s head. It is worth noting that " + hisOrHer()
                        + " fleet is unusually large and contains a unique Paladin of unknown origin. Use extreme caution when engaging.";
                break;
            }
            case STAGE_YAMATO: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-battlecruiser.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", was a renowned war veteran. For unknown reasons, " + heOrShe()
                        + " rebelled against the Sector's established governments. "
                        + person.getName().getLast() + " now brings his private fleet to bear against us. " + heOrSheCapital()
                        + " commands a unique Dominus, a famous ship that has seen many great battles over its long career.";
                break;
            }
            case STAGE_SPORESHIP: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 sporeship.";
                targetDescLong = "The target, " + person.getName().getFirst() + " " + person.getName().getLast()
                        + ", is an unknown AI entity, theorized to be a cluster of Alphas working in tandem. " + heOrSheCapital()
                        + " has somehow obtained a Domain-era sporeship, a legendary type of vessel, the last of which was thought"
                        + " decommissioned long ago. " + person.getName().getLast() + "'s intentions are unknown at this point, "
                        + "but " + heOrShe() + " appears to be building up an enormous fleet of automated ships. This development"
                        + " is a threat to the Persean Sector at large; the AI known as " + person.getName().getFirst() + " "
                        + person.getName().getLast() + " must be destroyed.";
                break;
            }
            case STAGE_POPE: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 1 mod-interdictor.";
                targetDescLong = "We have no information on the target, " + person.getName().getFirst() + " "
                        + person.getName().getLast()
                        + ", other than " + hisOrHer() + " current whereabouts. The presumption is that " + heOrShe()
                        + " is aligned with the Knights Templar in some way, judging by " + person.getName().getLast()
                        + "'s armor. Beware of " + hisOrHer()
                        + " extremely large fleet of cult followers, which is helmed by a unique Archbishop of immense power.";
                break;
            }
            case STAGE_ZEUS: {
                targetDesc = person.getName().getLast() + " commands a " + fleetSizeString()
                        + " fleet, including 2 mod-battleships, 1 mod-battlecruiser, and 4 mod-frigates.";
                targetDescLong = "PRIORITY ALERT: IBB founder " + person.getName().getFirst() + " "
                        + person.getName().getLast()
                        + " is AWOL. All available forces are asked to engage " + hisOrHer()
                        + " fleet posthaste. Before going rampant, " + heOrShe()
                        + " obtained a unique Paragon, a unique Conquest, and a unique Onslaught. Be advised: "
                        + person.getName().getLast()
                        + " is one the greatest living tacticians, and " + heOrShe()
                        + " is one of the Sector's foremost masters of starship design and modification. "
                        + heOrSheCapital()
                        + " is also in command of one of the largest, most advanced fleets the Sector has ever seen. Godspeed, bounty hunters.";
                break;
            }
            default:
                ended = true;
                break;
        }
    }

    @Override
    public boolean isDone() {
        return ended;
    }

    @Override
    public void reportBattleOccurred(CampaignFleetAPI primaryWinner, BattleAPI battle) {
        if (!isEventStarted()) {
            return;
        }
        if (isDone()) {
            return;
        }

        if (battle.isInvolved(fleet) && !battle.isPlayerInvolved()) {
            if (fleet.getFlagship() == null || fleet.getFlagship().getCaptain() != person) {
                fleet.setCommander(fleet.getFaction().createRandomPerson());
                Global.getSector().reportEventStage(this, "taken_end", market.getPrimaryEntity(), messagePriority);
                endEvent(false);
                return;
            }
        }

        CampaignFleetAPI playerFleet = Global.getSector().getPlayerFleet();
        if (!battle.isPlayerInvolved() || !battle.isInvolved(fleet) || battle.onPlayerSide(fleet)) {
            return;
        }

        // didn't destroy the original flagship
        if (fleet.getFlagship() != null && fleet.getFlagship().getCaptain() == person) {
            return;
        }

        RepLevel repLevel = playerFleet.getFaction().getRelationshipLevel(market.getFaction());
        if (!SWPModPlugin.isExerelin) {
            for (String ibbFaction : IBB_FACTIONS) {
                FactionAPI memberFac;
                try {
                    memberFac = Global.getSector().getFaction(ibbFaction);
                } catch (Exception e) {
                    continue;
                }
                if (memberFac != null) {
                    if (playerFleet.getFaction().getRelationshipLevel(memberFac).ordinal() > repLevel.ordinal()) {
                        repLevel = playerFleet.getFaction().getRelationshipLevel(memberFac);
                    }
                }
            }
        }

        payment = (int) (bountyCredits * battle.getPlayerInvolvementFraction());
        if (payment <= 0) {
            return;
        }

        completed = true;
        if (SWPModPlugin.isExerelin || repLevel.isAtWorst(RepLevel.SUSPICIOUS)) {
            String reportId = "bounty_payment_end";
            if (battle.getPlayerInvolvementFraction() < 1) {
                reportId = "bounty_payment_share_end";
            }
            log.info(String.format("Paying bounty of %f from the International Bounty Board", payment));
            Global.getSector().reportEventStage(this, reportId, market.getPrimaryEntity(),
                    MessagePriority.ENSURE_DELIVERY,
                    new BaseOnMessageDeliveryScript() {
                @Override
                public void beforeDelivery(CommMessageAPI message) {
                    CampaignFleetAPI playerFleet
                            = Global.getSector().getPlayerFleet();
                    playerFleet.getCargo().getCredits().add(payment);
                }
            });
        } else {
            log.info(String.format("Not paying bounty from the International Bounty Board"));
            Global.getSector().reportEventStage(this, "bounty_no_payment_end", market.getPrimaryEntity(),
                    MessagePriority.ENSURE_DELIVERY);
        }

        /* Mark bounty as completed */
        endEvent(false);
    }

    @Override
    public void reportFleetDespawned(CampaignFleetAPI fleet, FleetDespawnReason reason, Object param) {
        if (isDone()) {
            return;
        }

        if (this.fleet == fleet) {
            fleet.setCommander(fleet.getFaction().createRandomPerson());
            Global.getSector().reportEventStage(this, "other_end", market.getPrimaryEntity(), messagePriority);
            endEvent(false);
        }
    }

    @Override
    public void startEvent() {
        super.startEvent();
        if (person == null) {
            log.error("Failed to start event: no person");
            endEvent(true);
            return;
        }

        faction = market.getFaction();
        eventTarget = new CampaignEventTarget(market);

        if (isDone()) {
            log.error("Failed to start event: ended itself prematurely");
            return;
        }

        pickHideoutLocation();

        if (isDone()) {
            log.error("Failed to start event: hideout location invalid");
            return;
        }

        spawnFleet();

        SWP_IBBTracker.getTracker().reportStageBegan(thisStage);

        Global.getSector().reportEventStage(this, "start", null, messagePriority, new BaseOnMessageDeliveryScript() {
            @Override
            public void beforeDelivery(CommMessageAPI message) {
                if (hideoutLocation.getContainingLocation() instanceof StarSystemAPI) {
                    message.setStarSystemId(hideoutLocation.getContainingLocation().getId());
                }
            }
        });

        log.info(String.format("Starting famous bounty for person %s", person.getName().getFullName()));
    }

    private String fleetSizeString() {
        int pts = thisStage.maxPts + thisStage.ptsFromBoss;
        if (pts < 10) {
            return "very small";
        } else if (pts < 20) {
            return "small";
        } else if (pts < 40) {
            return "medium";
        } else if (pts < 60) {
            return "large";
        } else if (pts < 90) {
            return "very large";
        } else if (pts < 120) {
            return "huge";
        } else {
            return "gigantic";
        }
    }

    private String heOrShe() {
        if (it) {
            return "it";
        } else if (person.getGender() == Gender.MALE) {
            return "he";
        } else {
            return "she";
        }
    }

    private String heOrSheCapital() {
        if (it) {
            return "It";
        } else if (person.getGender() == Gender.MALE) {
            return "He";
        } else {
            return "She";
        }
    }

    private String himOrHer() {
        if (it) {
            return "it";
        } else if (person.getGender() == Gender.MALE) {
            return "him";
        } else {
            return "her";
        }
    }

    private String himOrHerCapital() {
        if (it) {
            return "It";
        } else if (person.getGender() == Gender.MALE) {
            return "Him";
        } else {
            return "Her";
        }
    }

    private String hisOrHer() {
        if (it) {
            return "its";
        } else if (person.getGender() == Gender.MALE) {
            return "his";
        } else {
            return "her";
        }
    }

    private String hisOrHerCapital() {
        if (it) {
            return "Its";
        } else if (person.getGender() == Gender.MALE) {
            return "His";
        } else {
            return "Her";
        }
    }

    private void pickHideoutLocation() {
        float emptyWeight = 1f;
        float occupiedWeight = 0.2f;

        float multMod = 1f;
        while (hideoutLocation == null && multMod <= 5f) {
            WeightedRandomPicker<StarSystemAPI> systemPicker = new WeightedRandomPicker<>();
            for (StarSystemAPI system : Global.getSector().getStarSystems()) {
                float weight = system.getPlanets().size();
                float mult = 1f;

                if (system.hasPulsar()) {
                    continue;
                }

                if (system.hasTag(Tags.THEME_MISC_SKIP)) {
                    mult = 1f;
                } else if (system.hasTag(Tags.THEME_MISC)) {
                    mult = 3f;
                } else if (system.hasTag(Tags.THEME_RUINS)) {
                    mult = 7f;
                } else if (system.hasTag(Tags.THEME_REMNANT_DESTROYED)) {
                    mult = 3f;
                }

                boolean viablePlanets = false;
                for (SectorEntityToken planet : system.getPlanets()) {
                    if (planet.isStar()) {
                        continue;
                    }
                    MarketAPI nearestMarket = findNearestMarket(planet, 3000f);
                    if (nearestMarket != null) {
                        continue;
                    }
                    viablePlanets = true;
                    break;
                }
                if (!viablePlanets) {
                    continue;
                }

                boolean empty = true;
                for (SectorEntityToken planet : system.getPlanets()) {
                    MarketAPI nearestMarket = findNearestMarket(planet, 3000f);
                    if (nearestMarket != null) {
                        continue;
                    }

                    if (planet.getMarket() != null && !planet.getMarket().isPlanetConditionMarketOnly()) {
                        empty = false;
                        break;
                    }
                }

                if (empty) {
                    mult *= emptyWeight;
                } else {
                    mult *= occupiedWeight;
                }

                if (mult <= 0) {
                    continue;
                }
                float minDist = 3000f + (thisStage.ordinal() + 1) * 200f;
                float maxDist = 30000f + (thisStage.ordinal() + 1) * 1000f;
                maxDist *= multMod;
                minDist /= multMod;
                float dist = system.getLocation().length();
                float distMult = (Math.max(0, maxDist - dist) / maxDist) * (Math.max(0, dist - minDist) / minDist);

                systemPicker.add(system, weight * mult * distMult);
            }

            StarSystemAPI system = systemPicker.pick();

            if (system != null) {
                WeightedRandomPicker<SectorEntityToken> picker = new WeightedRandomPicker<>();
                for (SectorEntityToken planet : system.getPlanets()) {
                    if (planet.isStar()) {
                        continue;
                    }
                    MarketAPI nearestMarket = findNearestMarket(planet, 3000f);
                    if (nearestMarket != null) {
                        continue;
                    }

                    picker.add(planet);
                }
                hideoutLocation = picker.pick();
            }

            multMod += 1f;
        }

        if (hideoutLocation == null) {
            log.info(String.format("Failed to find hideout for person %s", person.getName().getFullName()));
            endEvent(true);
        }
    }

    private void spawnFleet() {
        float qf = thisStage.qf;
        float commanderLevel = thisStage.level;
        float officerLevel = thisStage.officerLevel;
        int officerCount = thisStage.officerCount;
        int maxPts = thisStage.maxPts;
        int ptsFromBoss = thisStage.ptsFromBoss;
        String fleetFactionId = thisStage.fleetFaction;
        String fleetGenId = fleetFactionId;
        switch (fleetFactionId) {
            case "domain":
                fleetGenId = Factions.HEGEMONY;
                break;
            case "sector":
                fleetGenId = Factions.INDEPENDENT;
                break;
            case "everything":
                fleetGenId = Factions.LIONS_GUARD;
                break;
            default:
                break;
        }

        final FleetParams params = new FleetParams(hideoutLocation.getLocationInHyperspace(),
                market,
                fleetGenId,
                "famous_bounty",
                Math.max(maxPts, 1), // combatPts
                0f, // freighterPts
                0f, // tankerPts
                0f, // transportPts
                0f, // linerPts
                0f, // civilianPts
                0f, // utilityPts
                0f, // qualityBonus
                qf, // qualityOverride
                0f, // officer num mult
                0, // officer level bonus
                person,
                (int) commanderLevel);
        params.withOfficers = false;

        if (thisStage == FamousBountyStage.STAGE_SPORESHIP) {
            params.combatPts *= 0.67;
        }
        fleet = SWP_FleetFactory.enhancedCreateFleet(faction, Math.max(maxPts, 1) + ptsFromBoss,
                new FleetFactoryDelegate() {
            @Override
            public CampaignFleetAPI createFleet() {
                return FleetFactoryV2.createFleet(params);
            }
        });
        String fleetName = person.getName().getFirst() + " " + person.getName().getLast() + " (IBB)";
        fleet.setName(fleetName);

        flagship = Global.getFactory().createFleetMember(FleetMemberType.SHIP, thisStage.flagship);
        flagshipName = thisStage.flagshipName;
        person.setPersonality(thisStage.personality);

        fleet.getMemoryWithoutUpdate().set("$banterText", thisStage.banterText);

        /* Bounty definitions */
        List<FleetMemberAPI> specialShips = new ArrayList<>(10);
        switch (thisStage) {
            case STAGE_UNDINE: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "ssp_boss_lasher_r_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Salamander");
                specialShips.add(member);
                break;
            }
            case STAGE_HADES: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "ssp_boss_cerberus_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Dis");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_cerberus_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Sheol");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_cerberus_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Hell");
                specialShips.add(member);
                break;
            }
            case STAGE_PONY: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "ssp_boss_tarsus_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Carpal");
                specialShips.add(member);
                break;
            }
            case STAGE_FRACTURE: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "ssp_boss_hammerhead_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Phillipshead");
                specialShips.add(member);
                break;
            }
            case STAGE_RAPTOR: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "ssp_boss_medusa_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Gorgon");
                specialShips.add(member);
                break;
            }
            case STAGE_BULLSEYE: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "pack_bulldog_bullseye_Bullseye");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Charlie");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "pack_pitbull_bullseye_Bullseye");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Buddy");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "pack_pitbull_bullseye_Bullseye");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Rocky");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "pack_komondor_bullseye_Bullseye");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Jake");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "pack_komondor_bullseye_Bullseye");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Jack");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "pack_schnauzer_bullseye_Bullseye");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Toby");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "pack_schnauzer_bullseye_Bullseye");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Cody");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "pack_schnauzer_bullseye_Bullseye");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Buster");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "pack_schnauzer_bullseye_Bullseye");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Duke");
                specialShips.add(member);
                break;
            }
            case STAGE_EMPEROR: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "ssp_boss_brawler_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Boxer");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_brawler_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Fighter");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_brawler_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Fencer");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_brawler_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Wrestler");
                specialShips.add(member);
                break;
            }
            case STAGE_FRAMEBREAKER: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "ssp_boss_dominator_luddic_path_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Ludd's Hammer");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "ssp_boss_dominator_luddic_path_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Ludd's Wrath");
                specialShips.add(member);
                break;
            }
            case STAGE_LEVIATHAN: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "ssp_boss_phaeton_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Clymene");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_phaeton_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Tethys");
                specialShips.add(member);
                break;
            }
            case STAGE_ILIAD: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "ssp_boss_aurora_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Nebula");
                specialShips.add(member);
                break;
            }
            case STAGE_OLYMPUS_X: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "ii_boss_praetorian_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Praefectum");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ii_boss_praetorian_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Secundus");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ii_boss_praetorian_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Tertius");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ii_boss_praetorian_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Quartus");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ii_boss_praetorian_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Quintus");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ii_boss_praetorian_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Sextus");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ii_boss_praetorian_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Septimus");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ii_boss_praetorian_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Octavus");
                specialShips.add(member);
                break;
            }
            case STAGE_LUCIFER: {
                fleet.getFleetData().clear();

                FleetMemberAPI member
                        = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_doom_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Baphomet");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_euryale_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Euryale");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_euryale_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Moloch");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_euryale_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Asmodeus");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_euryale_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Mammon");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_afflictor_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Tormentor");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_afflictor_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Iblis");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_afflictor_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Dagon");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_afflictor_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Beelzebub");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_shade_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Phantom");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_shade_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Azazel");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_shade_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Apollyon");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_shade_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Samael");
                specialShips.add(member);
                break;
            }
            case STAGE_ODIN: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "msp_boss_potniaBis_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Maleficent");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "msp_boss_potniaBis_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Malevolent");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "msp_boss_potniaBis_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Malicious");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "msp_boss_potniaBis_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Nefarious");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "msp_boss_potniaBis_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Vicious");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "msp_boss_potniaBis_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Wicked");
                specialShips.add(member);
                break;
            }
            case STAGE_SPORESHIP: {
                params.factionId = Factions.DERELICT;
                CampaignFleetAPI temp = SWP_FleetFactory.enhancedCreateFleet(faction, Math.max(maxPts, 1) + ptsFromBoss,
                        new FleetFactoryDelegate() {
                    @Override
                    public CampaignFleetAPI createFleet() {
                        return FleetFactoryV2.createFleet(
                                params);
                    }
                });
                for (FleetMemberAPI member : temp.getFleetData().getMembersListCopy()) {
                    fleet.getFleetData().addFleetMember(member);
                }
                break;
            }
            case STAGE_ZEUS: {
                FleetMemberAPI member = Global.getFactory().createFleetMember(FleetMemberType.SHIP,
                        "ssp_boss_conquest_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Nike");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_onslaught_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Ares");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_hyperion_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Helios");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_hyperion_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Solar");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_hyperion_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Planetar");
                specialShips.add(member);

                member = Global.getFactory().createFleetMember(FleetMemberType.SHIP, "ssp_boss_hyperion_cus");
                fleet.getFleetData().addFleetMember(member);
                member.setShipName("Divinity");
                specialShips.add(member);
                break;
            }
            default: {
                break;
            }
        }

        FleetCompositionDoctrineAPI doctrine = faction.getCompositionDoctrine();
        float minLevel = officerLevel;
        minLevel -= minLevel * doctrine.getOfficerLevelVariance() * 0.5f;
        minLevel = (int) minLevel;
        if (minLevel < 1) {
            minLevel = 1;
        }

        float maxLevel = minLevel + minLevel * doctrine.getOfficerLevelVariance();
        maxLevel = (int) maxLevel;
        if (maxLevel < minLevel) {
            maxLevel = minLevel;
        }

        flagship.setShipName(flagshipName);
        fleet.getFleetData().addFleetMember(flagship);
        fleet.getFleetData().setFlagship(flagship);
        fleet.setCommander(person);
        flagship.setCaptain(person);
        fleet.setTransponderOn(true);
        fleet.removeAbility(Abilities.GO_DARK);
        fleet.removeAbility(Abilities.TRANSPONDER);

        fleet.forceSync();

        SWP_FleetFactory.addCommanderAndOfficersFamous(officerCount, minLevel, maxLevel, fleet, specialShips,
                new Random());

        fleet.forceSync();

        /* We do this deferred level-up so that the commander's skills reflect the flagship type */
        float weight = FleetFactoryV2.getMemberWeight(fleet.getFlagship());
        float fighters = fleet.getFlagship().getVariant().getFittedWings().size();
        boolean wantCarrierSkills = weight > 0 && fighters / weight >= 0.5f;
        SkillPickPreference pref = SkillPickPreference.NON_CARRIER;
        if (wantCarrierSkills) {
            pref = SkillPickPreference.CARRIER;
        }
        SWP_FleetFactory.levelOfficer(person, fleet.getFleetData(), thisStage.level, true, pref, new Random());

        /* Add in the commander skills, at last */
        FleetFactoryV2.addCommanderSkills(person, fleet, new Random());

        person.setRankId(Ranks.SPACE_ADMIRAL);
        person.setPostId(Ranks.POST_FLEET_COMMANDER);

        fleet.getFleetData().sort();
        fleet.forceSync();
        fleet.updateCounts();

        Map<String, String> archetypeOverride = new HashMap<>(specialShips.size());
        for (FleetMemberAPI member : specialShips) {
            archetypeOverride.put(member.getId(), "ULTIMATE");
        }

        List<String> noRandomize = new ArrayList<>(1);
        noRandomize.add(flagship.getId());

        Misc.makeImportant(fleet, "swp_ibb", duration);
        fleet.getMemoryWithoutUpdate().set("$level", commanderLevel);
        fleet.getMemoryWithoutUpdate().set("$opBonus", thisStage.opBonus);
        fleet.getMemoryWithoutUpdate().set(MemFlags.MEMORY_KEY_FLEET_TYPE, "famousBounty");
        if (thisStage == FamousBountyStage.STAGE_SPORESHIP) {
            fleet.getMemoryWithoutUpdate().set("$famousBountyFaction", "everything");
        } else {
            fleet.getMemoryWithoutUpdate().set("$famousBountyFaction", fleetFactionId);
        }
        fleet.getMemoryWithoutUpdate().set("$stillAlive", true);
        fleet.getMemoryWithoutUpdate().set("$ds_archetype_override", archetypeOverride);
        fleet.getMemoryWithoutUpdate().set("$ds_do_not_randomize", noRandomize);
        fleet.getMemoryWithoutUpdate().set("$stageName", thisStage.name());
        fleet.getMemoryWithoutUpdate().set(MemFlags.MEMORY_KEY_MAKE_AGGRESSIVE, true);
        fleet.getMemoryWithoutUpdate().set(MemFlags.FLEET_INTERACTION_DIALOG_CONFIG_OVERRIDE_GEN,
                new IBBInteractionConfigGen());

        fleet.setNoFactionInName(true);
        fleet.setFaction(Factions.NEUTRAL);

        FleetFactory.finishAndSync(fleet);
        LocationAPI location = hideoutLocation.getContainingLocation();
        location.addEntity(fleet);
        fleet.setLocation(hideoutLocation.getLocation().x - 500, hideoutLocation.getLocation().y + 500);
        fleet.getAI().addAssignment(FleetAssignment.ORBIT_AGGRESSIVE, hideoutLocation, 1000000f, null);
    }

    PersonAPI initPerson(FactionAPI bountyFaction) {
        PersonAPI bountyPerson = OfficerManagerEvent.createOfficer(bountyFaction, 1, true);
        bountyPerson.getName().setFirst(thisStage.firstName);
        bountyPerson.getName().setLast(thisStage.lastName);
        bountyPerson.getName().setGender(thisStage.gender);
        try {
            bountyPerson.setPortraitSprite(thisStage.portrait);
        } catch (Exception e) {
        }
        return bountyPerson;
    }

    public static enum FamousBountyStage {

        STAGE_UNDINE("Francis", "Butler", Gender.MALE, false, "graphics/swp/portraits/ibb_butler.png",
                RequiredFaction.NONE, 52500,
                "ssp_boss_lasher_b_cus", "Undine", "reckless",
                0.5f, 8, 6, 4, 8, 0f, 2, Factions.PIRATES,
                "So, you've come to claim our heads. You won't find us to be easy prey!"), // 0
        STAGE_HADES("Ryx", "Barlow", Gender.FEMALE, false, "graphics/swp/portraits/ibb_ryx.png",
                RequiredFaction.NONE, 79500,
                "ssp_boss_cerberus_cus", "Hades", "steady",
                0.5f, 12, 7, 4, 8, 0f, 4, Factions.INDEPENDENT,
                "These ships are mine. You'll never take them from me!"), // 1
        STAGE_PONY("Lisa", "Nanao", Gender.FEMALE, false, "graphics/swp/portraits/ibb_nanao.png",
                RequiredFaction.NONE, 105000,
                "ssp_boss_mule_cus", "Pony", "cautious",
                0.75f, 13, 8, 6, 12, 0f, 3, Factions.INDEPENDENT,
                "We're done playing games. Prepare to die!"), // 2
        STAGE_ELDER_ORB("Ocula", "Gazer", Gender.FEMALE, false, "graphics/swp/portraits/ibb_gazer.png",
                RequiredFaction.NONE, 140000,
                "ssp_boss_beholder_cus", "Elder Orb", "steady",
                0.75f, 14, 10, 7, 18, 0f, 2, Factions.DIKTAT,
                "Wow, would you look at the time? It's death o'clock!"), // 3
        STAGE_FRACTURE("Jaine", "Meredith", Gender.FEMALE, false, "graphics/swp/portraits/ibb_meredith.png",
                RequiredFaction.NONE, 175000,
                "ssp_boss_sunder_cus", "Fracture", "steady",
                0.5f, 15, 10, 6, 19, 0f, 4, Factions.LIONS_GUARD,
                "I hereby declare war against you, commander. En garde!"), // 4
        STAGE_LIBERTY("Abrahamo", "Lincolni", Gender.MALE, false,
                "graphics/tiandong/portraits/tiandong_boss_abrahamo_lincolni.png",
                RequiredFaction.TIANDONG, 160000,
                "tiandong_boss_wuzhang_Standard", "Liberty", "reckless",
                0.5f, 16, 11, 6, 13, 0f, 6, "tiandong",
                "You look like you could use an extra large serving of freedom!"), // 5
        STAGE_RAPTOR("Blaize", "Rex", Gender.FEMALE, false, "graphics/swp/portraits/ibb_rex.png",
                RequiredFaction.NONE, 235000,
                "ssp_boss_falcon_cus", "Raptor", "steady",
                0.75f, 15, 12, 8, 31, 0f, 6, Factions.INDEPENDENT,
                "Those are some nice ships you've got there. It would be a shame if something were to happen to them..."), // 6
        STAGE_APEX("Mackie", "Corbin", Gender.MALE, false, "graphics/swp/portraits/ibb_corbin.png",
                RequiredFaction.NONE, 220000,
                "ssp_boss_eagle_cus", "Apex", "aggressive",
                0.75f, 18, 13, 7, 21, 0f, 5, Factions.DIKTAT,
                "You're with THEM! The enemy! Call to arms, men! Make ready for war!"), // 7
        STAGE_BULLSEYE("Max", "Bailey", Gender.MALE, false, "graphics/portraits/portrait_hegemony06.png",
                RequiredFaction.JUNK_PIRATES, 300000,
                "pack_bulldog_bullseye_Bullseye", "Bullseye", "steady",
                0.75f, 16, 11, 10, 32, 0f, 14, "pack",
                "Woof woof, motherfucker!"), // 8
        STAGE_FRAMEBREAKER("Ned", "Ludd", Gender.MALE, false, "graphics/portraits/portrait_luddic00.png",
                RequiredFaction.NONE, 350000,
                "ssp_boss_onslaught_luddic_path_cus", "Framebreaker", "reckless",
                0.25f, 17, 13, 9, 48, 0f, 16, Factions.LUDDIC_PATH,
                "Lo, the Lord hath fortold this Apocalypse, a great event done unto us an aeon ago. And how the fires hath henceforth swept the "
                + "black-as-sin stars! Providence has brought another faithless heathen to me, here, today. The age of repentance is over; now, you "
                + "shall burn!"), // 9
        STAGE_EMPEROR("Johnny", "Major", Gender.MALE, false, "graphics/swp/portraits/ibb_major.png",
                RequiredFaction.NONE, 385000,
                "ssp_boss_dominator_cus", "Emperor", "aggressive",
                0.5f, 18, 14, 12, 52, 0f, 10, Factions.HEGEMONY,
                "You're here for me, too? Well, I won't go down without a fight! Come get some!"), // 10
        STAGE_GULF("Teresa", "Terror", Gender.FEMALE, false, "graphics/swp/portraits/ibb_terror.png",
                RequiredFaction.DIABLE, 415000,
                "diableavionics_IBBgulf_fang", "Gulf", "steady",
                0.75f, 20, 15, 11, 55, 0f, 6, "diableavionics",
                "You call THAT a fleet? Hah! I will crush you like a bug under my heel."), // 11
        STAGE_LEVIATHAN("Elaine", "Megas", Gender.FEMALE, false, "graphics/swp/portraits/ibb_megas.png",
                RequiredFaction.NONE, 465000,
                "ssp_boss_atlas_cus", "Leviathan", "cautious",
                0.25f, 19, 14, 15, 99, 0f, 10, Factions.PIRATES,
                "I was just trying to trade! This is bullshit!"), // 12
        STAGE_POSEIDON("Marjatta", "Nemo", Gender.FEMALE, false, "graphics/portraits/ms_portrait_001.png",
                RequiredFaction.SHADOWYARDS, 490000,
                "ms_boss_charybdis_cus", "Poseidon", "steady",
                1f, 20, 16, 12, 63, 0f, 6, "shadow_industry",
                "Do you have something I want? Hmm... Nah. I'll just blow you up."), // 13
        STAGE_ILIAD("Clade", "Midnight", Gender.MALE, false, "graphics/swp/portraits/ibb_midnight.png",
                RequiredFaction.NONE, 430000,
                "ssp_boss_odyssey_cus", "Iliad", "steady",
                1f, 22, 18, 8, 32, 0f, 12, Factions.TRITACHYON,
                "Sorry, commander. It's nothing personal, but I really want a good fight. Prepare yourself!"), // 14
        STAGE_OLYMPUS_X("Helmut", "Murkoph", Gender.MALE, false, "graphics/imperium/portraits/ii_portrait2.png",
                RequiredFaction.IMPERIUM, 570000,
                "ii_boss_olympus_cus", "Olympus-X", "timid",
                1f, 21, 19, 12, 56, 0f, 24, "interstellarimperium",
                "Bwahahahaha! Burn, baby, burn!"), // 15
        STAGE_FRANKENSTEIN("Xenno", "Aura", Gender.MALE, false, "graphics/swp/portraits/ibb_aura.png",
                RequiredFaction.NONE, 500000,
                "swp_boss_frankenstein_cus", "Frankenstein", "cautious",
                0.75f, 26, 20, 14, 63, 0f, 6, Factions.PIRATES,
                "Oops! XD"), // 16
        STAGE_ODIN("RA-002", "SHADOW", Gender.MALE, true, "graphics/portraits/ms_portrait_006.png",
                RequiredFaction.SHADOWYARDS, 660000,
                "ms_boss_mimir_cus", "Odin", "aggressive",
                0.5f, 23, 20, 14, 59, 0f, 20, "sector",
                "PURPOSE."), // 17
        STAGE_LUCIFER("TTX", "LaCROIX", Gender.FEMALE, true, "graphics/swp/portraits/ibb_lacroix.png",
                RequiredFaction.NONE, 570000,
                "ssp_boss_doom_cus", "Lucifer", "reckless",
                1f, 28, 25, 9, 0, 0f, 40, Factions.TRITACHYON,
                "I am sorry."), // 18
        STAGE_EUPHORIA("Snipe", "Gunzup", Gender.MALE, false, "graphics/portraits/portrait_corporate06.png",
                RequiredFaction.CABAL, 735000,
                "ssp_boss_astral_cus", "Euphoria", "steady",
                1f, 29, 23, 13, 56, 1f, 16, "cabal",
                "I am at a higher state of being than you. How could you possibly understand the truth?"), // 19
        STAGE_CLERIC("Mathis", "Adelier", Gender.MALE, false, "graphics/templars/portraits/tem_portrait1.png",
                RequiredFaction.TEMPLARS, 770000,
                "tem_boss_paladin_cus", "Cleric", "reckless",
                1f, 29, 21, 12, 50, 3f, 8, "templars",
                "The time of forbearance has finished. The hour of victory, of justice, has come. And now, you shall join the fire."), // 20
        STAGE_YAMATO("Katsu", "Okita", Gender.MALE, false, "graphics/imperium/portraits/ii_okita.png",
                RequiredFaction.IMPERIUM, 645000,
                "ii_boss_dominus_cus", "Yamato", "steady",
                0.75f, 29, 22, 16, 56, 4f, 8, "interstellarimperium",
                "You won't stop me from saving my people!"), // 21
        STAGE_POPE("Adam", "Tipper", Gender.MALE, true, "graphics/swp/portraits/ibb_tipper.png",
                RequiredFaction.TEMPLARS, 850000,
                "tem_boss_archbishop_cus", "Pope", "aggressive",
                1f, 29, 20, 18, 58, 5f, 24, Factions.LIONS_GUARD,
                "I. AM. ETERNAL."), // 22
        STAGE_SPORESHIP("El", "Psi", Gender.MALE, true, "graphics/swp/portraits/ibb_psi.png",
                RequiredFaction.NONE, 905000,
                "swp_boss_sporeship_cus", "DES S-02", "aggressive",
                1f, 29, 21, 20, 76, 7f, 24, Factions.REMNANTS,
                "You are not omega."), // 23
        STAGE_ZEUS("Gabriel", "Mosolov", Gender.MALE, false, "graphics/portraits/portrait_hegemony05.png",
                RequiredFaction.NONE, 1250000,
                "ssp_boss_paragon_cus", "Zeus", "aggressive",
                1f, 29, 24, 18, 85, 10f, 20, "everything",
                "It was nice of you to kill off all of my competitors for me. You have my sincerest thanks. Now, you die."); // 24

        public final String firstName;
        public final String lastName;
        public final Gender gender;
        public final boolean machine;
        public final String portrait;

        public final RequiredFaction mod;
        public final int reward;

        public final String flagship;
        public final String flagshipName;
        public final String personality;

        public final float qf;
        public final int level;
        public final int officerLevel;
        public final int officerCount;
        public final int maxPts;
        public final float opBonus;
        public final int ptsFromBoss;

        public final String fleetFaction;

        public final String banterText;

        private FamousBountyStage(String firstName, String lastName, Gender gender, boolean machine, String portrait,
                RequiredFaction mod, int reward, String flagship, String flagshipName,
                String personality, float qf, int level, int officerLevel, int officerCount,
                int maxPts, float opBonus, int ptsFromBoss, String fleetFaction, String banterText) {
            this.firstName = firstName;
            this.lastName = lastName;
            this.gender = gender;
            this.machine = machine;
            this.portrait = portrait;

            this.mod = mod;
            this.reward = reward;

            this.flagship = flagship;
            this.flagshipName = flagshipName;
            this.personality = personality;

            this.qf = qf;
            this.level = level;
            this.officerLevel = officerLevel;
            this.officerCount = officerCount;
            this.maxPts = maxPts;
            this.opBonus = opBonus;
            this.ptsFromBoss = ptsFromBoss;

            if (!SWPModPlugin.hasDynaSector) {
                switch (fleetFaction) {
                    case "domain":
                        this.fleetFaction = Factions.HEGEMONY;
                        break;
                    case "sector":
                        this.fleetFaction = Factions.INDEPENDENT;
                        break;
                    case "everything":
                        this.fleetFaction = Factions.LIONS_GUARD;
                        break;
                    default:
                        this.fleetFaction = fleetFaction;
                        break;
                }
            } else {
                this.fleetFaction = fleetFaction;
            }

            this.banterText = banterText;
        }
    }

    public static class IBBInteractionConfigGen implements FIDConfigGen {

        @Override
        public FIDConfig createConfig() {
            FIDConfig config = new FIDConfig();
            config.showWarningDialogWhenNotHostile = false;
            config.impactsEnemyReputation = false;
            config.showFleetAttitude = false;
            config.pullInAllies = false;
            config.pullInEnemies = false;

            config.delegate = new FIDDelegate() {
                @Override
                public void battleContextCreated(InteractionDialogAPI dialog, BattleCreationContext bcc) {
                    bcc.enemyDeployAll = true;
                }

                @Override
                public void notifyLeave(InteractionDialogAPI dialog) {
                }

                @Override
                public void postPlayerSalvageGeneration(InteractionDialogAPI dialog, FleetEncounterContext context,
                        CargoAPI salvage) {
                }
            };

            return config;
        }
    }
}
